
#ifndef _15_CE2_DH_h_H
#define _15_CE2_DH_h_H

/*
 * All the includes that are needed for code using this module to
 * compile correctly should be #included here.
 */

#include "CE2_public.h"

#ifdef __cplusplus
extern "C"
{
#endif

/*
 *  Object name     :  CE2_DH.h
 */

  /*
   *  Object %name    : %CE2_DH.h
   *  State           :  %state%
   *  Creation date   :  Wed Jan 26 16:54:02 2005
   *  Last modified   :  %modify_time%
   */
  /** @file
   *  \brief This module defines the API that supports Diffie-Hellman key exchange 
   *  as defined in PKCS#3 and ANSI9.42 v1.4 (Revised from November 1,1993)
   *
   *  \version CE2_DH.h#1:incl:15
   *  \author ohads
   *  \remarks Copyright (C) 2005 by Discretix Technologies Ltd.
   *           All Rights reserved
   */




/************************ Defines ******************************/
/************************ Enums ********************************/

typedef enum
{
   CE2_DH_PKCS3_mode  = 0,
   CE2_DH_ANSI_X942_mode = 1,
   CE2_DH_NumOfModes

}CE2_DH_OpMode_t;   


typedef enum
{
   CE2_DH_HASH_SHA1_mode  = 0,
   CE2_DH_HASH_NumOfModes

}CE2_DH_HASH_OpMode_t;   

typedef enum
{
   CE2_DH_ASN1_Der_mode  = 0,
   CE2_DH_Concat_Der_mode  = 1,   
   CE2_DH_der_NumOfModes

}CE2_DH_DerivationFunc_Mode;   

/************************ Typedefs  ****************************/


/************************ Structs  ******************************/


/************************ Public Variables **********************/


/************************ Public Functions **********************/

/**
****************************************************************
* Function Name: 
*  CE2_DX_DH_GeneratePubPrv
*
*  @param Generator_ptr [in] - Pointer to the Generator octet string
*  @param GeneratorSize [in] - Size of the Generator String (in bytes)
*  @param Prime_ptr [in] - Pointer to the Prime octet string
*  @param PrimeSize [in] - Size of the Prime string (in bytes)
*  @param L [in] - Relevant only for PKCS#3, the private value length in bits
*                 If L != 0 then - force the private key to be 2^(L-1) <= Prv < 2^l
*                 if L 0 then: 0< Prv < P-1
*                 In ANSI X9.42 L is irrelevant  
*  @param Q_ptr [in] - Relevant only for ANSI X9.42 - Pointer to the Q octet string
*                     1 <= Prv <= q-1   or   1 < Prv < q-1
*  @param QSize	[in] - Relevant only for ANSI X9.42 - Size of the Q string (in bytes)  				 				  
*  @param DH_mode [in] - enum declaring weather this is PKCS#3 or ANSI X9.42
*  @param ClientPrvKey_ptr [out] - Pointer to the Private key octet string - 
*                                 In PKCS#3 This buffer should be at least the 
*                                 following size:
*                                 2^L - 1 - if L is provided
*                                 P-1	  - if L is DX_NULL
*  @param ClientPrvKeySize_ptr [in/out] - The user should provide the size of the 
*                                        buffer indicated by ClientPrvKey_ptr
*                                        The function returns the actual size 
*                                        in bytes of the Private key Size
*  @param ClientPub_ptr [out] - Pointer to the Public key octet string
*                              This Buffer should be at least PrimeSize bytes
*  @param ClientPubSize_ptr [in/out] - The user should provide the size of the buffer 
*                                     indicated by ClientPub_ptr
*                                     The function returns the actual size in bytes 
*                                     of the client public buffer
*
* @returns \b
*  CE2Error_t 
*  - CE2_OK - on success
*  - Otherwise - error code defined in CE2_DH_error.h:
*
* \brief \b 
* Description:
*  _DX_DH_GeneratePubPrv has 2 functionalities:
*  -# randomly generates the Client private key 
*  -# compute the Client public key which is 
*     ClientPub = Generator ^ Prv mod Prime 
*
*  \b 
* Algorithm:
*  -# Verify input parameters for validity;
*  -# Call low level function LLF_DH_GeneratePubPrv 
*     for DH with using LibTomCrypt.
***************************************************************/
CE2CIMPORT_C CE2Error_t CE2_DX_DH_GeneratePubPrv(DxUint8_t *Generator_ptr,
				                    DxUint16_t GeneratorSize,
				                    DxUint8_t *Prime_ptr,
				                    DxUint16_t PrimeSize,
				                    DxUint16_t L,
				                    DxUint8_t *Q_ptr,
				                    DxUint16_t QSize,
				                    CE2_DH_OpMode_t DH_mode,
				                    DxUint8_t *ClientPrvKey_ptr,
				                    DxUint16_t *ClientPrvKeySize_ptr,
				                    DxUint8_t *ClientPub_ptr,
				                    DxUint16_t *ClientPubSize_ptr);



/*Q is irrelevant for PKCS#3 */
#define CE2_DH_PKCS3_GeneratePubPrv(Generator_ptr,GeneratorSize,\
									Prime_ptr,PrimeSize,\
									L,\
									ClientPrvKey_ptr,ClientPrvKeySize_ptr,\
									ClientPub_ptr,ClientPubSize_ptr)\
		CE2_DX_DH_GeneratePubPrv(Generator_ptr,GeneratorSize,\
								Prime_ptr,PrimeSize,\
								L,\
								DX_NULL,0,\
								CE2_DH_PKCS3_mode,\
								ClientPrvKey_ptr,ClientPrvKeySize_ptr,\
								ClientPub_ptr,ClientPubSize_ptr)

/*L is irrelevant for ANSI X9.42*/
#define CE2_DH_ANSI_X942_GeneratePubPrv(Generator_ptr,GeneratorSize,Prime_ptr,PrimeSize,\
									Q_ptr,QSize,\
									ClientPrvKey_ptr,ClientPrvKeySize_ptr,\
									ClientPub_ptr,ClientPubSize_ptr)\
		CE2_DX_DH_GeneratePubPrv(Generator_ptr,GeneratorSize,\
								Prime_ptr,PrimeSize,\
								0,\
								Q_ptr,QSize,\
								CE2_DH_ANSI_X942_mode,\
								ClientPrvKey_ptr,ClientPrvKeySize_ptr,\
								ClientPub_ptr,ClientPubSize_ptr)

/**
****************************************************************
* Function Name: 
*  CE2_DH_PKCS3_GetSecretKey
*
*  @param ClientPrvKey_ptr [in] - Pointer to the Private key octet string.
*  @param ClientPrvKeySize [in] - The Private key Size (in bytes)
*  @param ServerPubKey_ptr [in] - Pointer to the Server public key octet string.
*  @param ServerPubKeySize [in] - The Server Public key Size (in bytes)
*  @param Prime_ptr [in] - Pointer to the Prime octet string
*  @param PrimeSize [in] - Size of the Prime string
*  @param SecretKey_ptr [out] - Pointer to the secret key octet string.
*                              This buffer should be at least PrimeSize Bytes
*  @param SecretKeySize_ptr [out] - Size of secret key octet string
*
* @returns \b
*  CE2Error_t 
*  - CE2_OK - on success
*  - Otherwise - error code defined in CE2_DH_error.h:
*
* \brief \b 
* Description:
*  CE2_DH_PKCS3_GetSecretKey computes the shared secret key in the 
*  following computation: SecretKey = ServerPubKey ^ ClientPrvKey mod Prime
*
*  \b 
* Algorithm:
*  -# Verify input parameters for validity;
*  -# Call low level function LLF_DH_PKCS3_GetSecretKey. 
***************************************************************/
CE2CIMPORT_C CE2Error_t CE2_DH_PKCS3_GetSecretKey(DxUint8_t *ClientPrvKey_ptr,
				                    DxUint16_t ClientPrvKeySize,
 				                    DxUint8_t *ServerPubKey_ptr,
				                    DxUint16_t ServerPubKeySize,				                    
				                    DxUint8_t *Prime_ptr,
				                    DxUint16_t PrimeSize,
				                    DxUint8_t *SecretKey_ptr,
				                    DxUint16_t *SecretKeySize_ptr);


/**
****************************************************************
* Function Name: 
*  _DX_DH_KeyDerivationFunc
*
*  @param ZZSecret_ptr [in] - A pointer to shared secret key octet string 
*  @param ZZSecretSize [in] - The shared secret key Size, in bytes
*  @param hashFunc [in] - The hash function to be used. The hash function 
*                        output must be at least 160 bits.
*  @param KeyLenInBits	[in] - The size in Bits of the keying data to be generated - 
*                             KeyLenInBits < hashlen*(2^32-1) - in our implementation 
*                             the size should not be larger than the maximum input 
*                             size to the Hash function
*  @param derivation_mode [in] - Specifies whether we use an ASN.1-based derivation 
*                               function or a derivation based on concatenation.
*  @param KeyingData_ptr [out] - A pointer to the keying data derived from the secret 
*                               key, of length KeyLenInBits
*
* @returns \b
*  CE2Error_t 
*  - CE2_OK - on success
*  - Otherwise - error code defined in CE2_DH_error.h:
*
* \brief \b 
* Description:
*  _DX_DH_KeyDerivationFunc specifies the key derivation function, 
*  either based on ASN.1 DER encoding or based on the concatanation of fields as 
*  specified in ANSI X9.42-2001. The purpose of this function is to derive a key 
*  data from the shared secret value.
*
*  \b 
* Algorithm:
*  -# Verify input parameters for validity;
*  -# Call low level function LLF_DH_KeyDerivationFunc. 
***************************************************************/
CE2CIMPORT_C CE2Error_t CE2_DX_DH_KeyDerivationFunc(DxUint8_t *ZZSecret_ptr,
				                      DxUint16_t ZZSecretSize,
				  				      CE2_DH_HASH_OpMode_t hashFunc,				                    
				                      DxUint32_t KeyLenInBits,				                    
				                      DxUint8_t *KeyingData_ptr,
				                      CE2_DH_DerivationFunc_Mode derivation_mode);
				                    

				                    
#define CE2_DH_ASN1_KeyDerivation_SHA1Func(ZZSecret_ptr,ZZSecretSize,KeyLenInBits,KeyingData_ptr)\
		CE2_DX_DH_KeyDerivationFunc(ZZSecret_ptr,ZZSecretSize,CE2_DH_HASH_SHA1_mode,KeyLenInBits,KeyingData_ptr,CE2_DH_ASN1_Der_mode)
				                    

#define CE2_DH_Concat_KeyDerivation_SHA1Func(ZZSecret_ptr,ZZSecretSize,KeyLenInBits,KeyingData_ptr)\
		CE2_DX_DH_KeyDerivationFunc(ZZSecret_ptr,ZZSecretSize,CE2_DH_HASH_SHA1_mode,KeyLenInBits,KeyingData_ptr,CE2_DH_Concat_Der_mode)


/**
****************************************************************
* Function Name: 
*  CE2_DX_DH_X942_GetSecretData
*
*  @param hashFunc [in] - The hash function to be used
*  @param ClientPrvKey_ptr [in] - A pointer to the Private key octet string 
*  @param ClientPrvKeySize [in] - The Private key size, in bytes
*  @param ServerPubKey_ptr [in] - A pointer to the Server public key octet string - 
*  @param ServerPubKeySize [in] - The Server Public key size, in bytes
*  @param Prime_ptr [in] - A pointer to the Prime octet string
*  @param PrimeSize [in] - The size of the Prime string
*  @param DerFunc_mode [in] - The type of function to derive the secret key to 
*                            the key data. We use ASN.1-based function or Hash 
*                            concatenation function mode.
*  @param SecretKeyData_ptr [out] - A pointer to the secret key octet string. 
*                                  This buffer should be at least PrimeSize bytes.
*  @param SecretKeyDataSizeBitsNeeded [in] - Specifies the derived Secret Key data 
*                                           size needed in Bits. This value converted 
*                                           to bytes cannot be larger than PrimeSize
*
* @returns \b
*  CE2Error_t 
*  - CE2_OK - on success
*  - Otherwise - error code defined in CE2_DH_error.h:
*
* \brief \b 
* Description:
*  _DX_DH_X942_GetSecretData computes the shared secret key as follows:
*   -# OPTIONAL - validate the correctness of the argument keys
*   -# SecretKey = ServerPubKey ^ ClientPrvKey mod Prime
*   -# Use of Derivation function to derive a data key from the secret key 
*
*  \b 
* Algorithm:
*  -# Verify input parameters for validity;
*  -# Call low level function LLF_DH_KeyDerivationFunc. 
***************************************************************/
CE2CIMPORT_C CE2Error_t CE2_DX_DH_X942_GetSecretData(
				  				          CE2_DH_HASH_OpMode_t hashFunc,		
									          DxUint8_t *ClientPrvKey_ptr,
				                    DxUint16_t ClientPrvKeySize,
 				                    DxUint8_t *ServerPubKey_ptr,
				                    DxUint16_t ServerPubKeySize,				                    
				                    DxUint8_t *Prime_ptr,
				                    DxUint16_t PrimeSize,
				                    CE2_DH_DerivationFunc_Mode DerFunc_mode,
				                    DxUint8_t *SecretKeyData_ptr,
				                    DxUint16_t SecretKeyDataSizeBitsNeeded);

/**
   \brief CE2_DH_X942_GetSecret_ASN1_2_Data implements the DH ANSI X9.42 standard. It derives a secret key 
   			using the Derivation function based on ASN1
*/
#define CE2_DH_X942_GetSecret_ASN1_2_Data(hashFunc,ClientPrvKey_ptr,ClientPrvKeySize,ServerPubKey_ptr,ServerPubKeySize,Prime_ptr,PrimeSize,SecretKeyData_ptr,SecretKeyDataSizeBitsNeeded)\
	CE2_DX_DH_X942_GetSecretData(hashFunc,ClientPrvKey_ptr,ClientPrvKeySize,ServerPubKey_ptr,ServerPubKeySize,Prime_ptr,PrimeSize,CE2_DH_ASN1_Der_mode,SecretKeyData_ptr,SecretKeyDataSizeBitsNeeded)
/**
   \brief CE2_DH_X942_GetSecret_Concat_2_Data implements the DH ANSI X9.42 standard. It derives a secret key 
   			using the Derivation function based on concatenation of SHA1 hash 
*/				                    
#define CE2_DH_X942_GetSecret_Concat_2_Data(hashFunc,ClientPrvKey_ptr,ClientPrvKeySize,ServerPubKey_ptr,ServerPubKeySize,Prime_ptr,PrimeSize,SecretKeyData_ptr,SecretKeyDataSizeBitsNeeded)\
	CE2_DX_DH_X942_GetSecretData(hashFunc,ClientPrvKey_ptr,ClientPrvKeySize,ServerPubKey_ptr,ServerPubKeySize,Prime_ptr,PrimeSize,CE2_DH_Concat_Der_mode,SecretKeyData_ptr,SecretKeyDataSizeBitsNeeded)
				                    
/**
****************************************************************
* Function Name: 
*  _DX_DH_X942_Hybrid_GetSecretData
*
*  @param hashFunc [in] - The hash function to be used
*  @param ClientPrvKey_ptr1 [in] - A pointer to the First Private key octet string 
*                                 number 
*  @param ClientPrvKeySize1 [in] - The First Private key Size, in bytes
*  @param ClientPrvKey_ptr2 [in] - A pointer to the Second Private key octet string
*  @param ClientPrvKeySize2 [in] - The Second Private key Size, in bytes
*  @param ServerPubKey_ptr1 [in] - A pointer to the First Server public key octet string
*  @param ServerPubKeySize1 [in] - The First Server Public key Size, in bytes
*  @param ServerPubKey_ptr2 [in] - A pointer to the Second Server public key octet string
*  @param ServerPubKeySize2 [in] - The Second Server Public key Size, in bytes
*  @param Prime_ptr [in] - A pointer to the Prime octet string
*  @param PrimeSize [in] - The size of the Prime string
*  @param DerFunc_mode [in] - The type of function to derive the secret key to the key 
*                            data. We use an ASN.1-based function or a Hash 
*                            concatenation function mode.
*  @param SecretKeyData_ptr [out] - A pointer to the secret key octet string.
*                                  This buffer should be at least 2*PrimeSize bytes.
*  @param SecretKeyDataSizeBitsNeeded [in] - Specifies the derived Secret Key data size 
*                                           needed in Bits. This value converted to bytes 
*                                           cannot be larger than PrimeSize
*
* @returns \b
*  CE2Error_t 
*  - CE2_OK - on success
*  - Otherwise - error code defined in CE2_DH_error.h:
*
* \brief \b 
* Description:
*  _DX_DH_X942_Hybrid_GetSecretData computes the shared secret key as follows:
*   -# OPTIONAL - validate the correctness of the argument keys
*   -# SecretKey1 = ServerPubKey1 ^ ClientPrvKey1 mod Prime
*   -# SecretKey2 = ServerPubKey2 ^ ClientPrvKey2 mod Prime
*   -# Use of Derivation function to derive a data key from the 2 secret keys 
*
*  \b 
* Algorithm:
*  -# Verify input parameters for validity;
*  -# Call low level function LLF_DH_X942_Hybrid_GetSecretData. 
***************************************************************/
CE2CIMPORT_C CE2Error_t CE2_DX_DH_X942_Hybrid_GetSecretData(
      				  				    CE2_DH_HASH_OpMode_t hashFunc,		
 									          DxUint8_t *ClientPrvKey_ptr1,
				                    DxUint16_t ClientPrvKeySize1,
 									          DxUint8_t *ClientPrvKey_ptr2,
				                    DxUint16_t ClientPrvKeySize2,				                    
 				                    DxUint8_t *ServerPubKey_ptr1,
				                    DxUint16_t ServerPubKeySize1,
 				                    DxUint8_t *ServerPubKey_ptr2,
				                    DxUint16_t ServerPubKeySize2,				                    				                    
				                    DxUint8_t *Prime_ptr,
				                    DxUint16_t PrimeSize,
				                    CE2_DH_DerivationFunc_Mode DerFunc_mode,
				                    DxUint8_t *SecretKeyData_ptr,
				                    DxUint16_t SecretKeyDataSizeBitsNeeded);

/**
   \brief CE2_DH_X942_Hybrid_GetSecret_ASN1_2_Data implements the DH ANSI X9.42 standard deriving a hybrid secret key 
   			from 2 public-private pair of keys using the Derivation function based on ASN1
*/
#define CE2_DH_X942_Hybrid_GetSecret_ASN1_2_Data(hashFunc,ClientPrvKey_ptr1,ClientPrvKeySize1,ClientPrvKey_ptr2,ClientPrvKeySize2,ServerPubKey_ptr1,ServerPubKeySize1,ServerPubKey_ptr2,ServerPubKeySize2,Prime_ptr,PrimeSize,SecretKeyData_ptr,SecretKeyDataSizeBitsNeeded)\
		CE2_DX_DH_X942_Hybrid_GetSecretData(hashFunc,ClientPrvKey_ptr1,ClientPrvKeySize1,ClientPrvKey_ptr2,ClientPrvKeySize2,ServerPubKey_ptr1,ServerPubKeySize1,ServerPubKey_ptr2,ServerPubKeySize2,Prime_ptr,PrimeSize,CE2_DH_ASN1_Der_mode,SecretKeyData_ptr,SecretKeyDataSizeBitsNeeded)
				                    			  
/**
   \brief CE2_DH_X942_Hybrid_GetSecret_Concat_2_Data implements the DH ANSI X9.42 standard deriving a hybrid secret key 
   			from 2 public-private pair of keys using the Derivation function based on conctenation using SHA-1 Hash
*/				                    			  
#define CE2_DH_X942_Hybrid_GetSecret_Concat_2_Data(hashFunc,ClientPrvKey_ptr1,ClientPrvKeySize1,ClientPrvKey_ptr2,ClientPrvKeySize2,ServerPubKey_ptr1,ServerPubKeySize1,ServerPubKey_ptr2,ServerPubKeySize2,Prime_ptr,PrimeSize,SecretKeyData_ptr,SecretKeyDataSizeBitsNeeded)\
        CE2_DX_DH_X942_Hybrid_GetSecretData(hashFunc,ClientPrvKey_ptr1,ClientPrvKeySize1,ClientPrvKey_ptr2,ClientPrvKeySize2,ServerPubKey_ptr1,ServerPubKeySize1,ServerPubKey_ptr2,ServerPubKeySize2,Prime_ptr,PrimeSize,CE2_DH_Concat_Der_mode,SecretKeyData_ptr,SecretKeyDataSizeBitsNeeded)
                             
                             
#ifdef __cplusplus
}
#endif

#endif
